<?php
// auto_settings.php  -- single-file UI + AJAX handler (writes /var/www/html/ramdisk/sets.csv)

// Paths 
$dir = "/var/www/html/ramdisk/";
$statusFile = $dir . "status.csv";
$setsFile   = $dir . "sets.csv";
$ackFile    = $dir . "sets.ack";
$rlyFile    = "/var/www/html/CalFiles/rly.csv";

// ---------- Read status.csv into $statusValues (used to populate form defaults) ----------
$statusData = [];
if (file_exists($statusFile)) {
    $statusData = file($statusFile, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
}

$statusValues = [];
if (!empty($statusData)) {
    foreach ($statusData as $line) {
        $lineValues = str_getcsv($line);
        $statusValues = array_merge($statusValues, $lineValues);
    }
}

// --- Display status.csv in HTML table ---
if (!empty($statusValues)) {
    echo "<h3>UREBS Status</h3>";
    // Enable Output (strip high nibble)
    $val = isset($statusValues[0]) ? ($statusValues[0] & 0x0F) : "";
    echo "Enable Output:" . $val . "  ";
    // Field Level
    $val = $statusValues[1] ?? "";
    echo " Level:" . $val . ", ";
    // Field Mode (numeric ASCII → char)
    $val = isset($statusValues[2]) ? chr((int)$statusValues[2]) : "";
    echo " Mode:" . $val . ", ";
    // Toff Value
    $val = $statusValues[3] ?? "";
    echo " Toff:" . $val . ", ";
    // TT on
    $val = $statusValues[4] ?? "";
    echo " TT on:" . $val . ", ";
    // TT off
    $val = $statusValues[5] ?? "";
    echo " TT off:" . $val . "<br>";
    // TBD3
    $val = $statusValues[6] ?? "";
    echo " TBD3" . $val . ", ";
    // Cal Mode
    $val = $statusValues[7] ?? "";
    echo "Cal Mode:" . $val ;
    // Cal Level
    $val = $statusValues[8] ?? "";
    echo $val . "<br>";
    // Phase Calibration
    $val = $statusValues[9] ?? "";
    echo " Phase Calibration:" . $val . "<br>";
        
    // flgtt
    $val = $statusValues[10] ?? "";
    echo " flgtt:" . $val . " ";
    // cntBursts
    $valM = $statusValues[11] ?? "";
    $valL = $statusValues[12] ?? "";
    echo " cntBursts:" . ($valM*256)+$valL . "<br>";
    // Timer Value
    $valM = $statusValues[13] ?? "";
    $valL = $statusValues[14] ?? "";
    $val = ($valM*256)+$valL;
    $curFreq = 24000000/($val+1)/2;   //calculate current burst freq based on the timer value
    echo " Timer:" .  $val  . " MSB:" . $valM . " LSB:" . $valL . "  Freq:" . number_format($curFreq,2) . " ";
    // N_CYCLES
    $valnCyc = $statusValues[15] ?? "";
    echo " #Cycles:" . $valnCyc . "<br>";
    
    // Vert Level (20 bytes)
    $rowValues = array_slice($statusValues, 40, 20);
    echo " Vert Level:" . htmlspecialchars(implode(",", $rowValues)) . "<br>";
    // Horz Level (20 bytes)
    $rowValues = array_slice($statusValues, 20, 20);
    echo " Horz Level:" . htmlspecialchars(implode(",", $rowValues)) . "<br>";


    // Phase (20 bytes) with signed char conversion
    $rowValues = array_slice($statusValues, 60, 20);
    foreach ($rowValues as &$v) {
        if ($v > 127) {
            $v -= 256; // convert unsigned byte → signed char
        }
        //apply correction factor based on frequency
        $v = $v / (1/$curFreq/360/10E-6);  //divid operation Ex: 210Hz actually has 476 10us ticks, so each tick is worth less than one degree
        $v = round($v);   // round to nearest degree
    }
    unset($v);
    echo " Phase:" . htmlspecialchars(implode(",", $rowValues)) . "<br>";

    echo "</table>";
}




// ---------- Handle AJAX POST updates (single-field updates) ----------
if ($_SERVER['REQUEST_METHOD'] === 'POST' && !empty($_POST)) {
    // Load existing sets.csv values if present, otherwise zero-fill 20 elements
    if (file_exists($setsFile)) {
        $setsValues = str_getcsv(file_get_contents($setsFile));
    } else {
        $setsValues = array_fill(0, 20, 0);
    }
    // ensure at least 20 slots
    for ($i = 0; $i < 20; $i++) {
        if (!isset($setsValues[$i])) $setsValues[$i] = 0;
    }

    // Update indices based on posted field name(s)
    foreach ($_POST as $key => $val) {
        switch ($key) {
            case "enOut":       $setsValues[0] = (int)$val; break; // element 0 (will OR 0xA0 later)
            case "f_level": $setsValues[1] = (int)$val; break;
            case "f_mode":  $setsValues[2] = (int)$val; break;
            case "toff":        $setsValues[3] = (int)$val; break;
            case "ttOn":        $setsValues[4] = (int)$val; break;
            case "ttOff":       $setsValues[5] = (int)$val; break;
            case "ttRst":       $setsValues[6] = (int)$val; break;
            case "cal":         $setsValues[7] = (int)$val; break;   // 0 / 1 / 99
            case "calFile":     $setsValues[8] = $val; break;        // string or numeric
            case "freq":
                $freqVal = (int)$val;
                $timer = round((24000000/($freqVal*2))-1); 
                $setsValues[9] = (int)($timer/256);
                $setsValues[10] = (int)($timer&0xFF);
                //set the relay values based on the freq
                //send the correct calibration file for the freq
                $setsValues[8] = 0x80;   //temporary override of the calFile value to force correct cal files set be sent to UREBS
                $allowed = [210,215,220,225,230,235,240,245,250,255,260,265];
                if (in_array($freqVal, $allowed, true)) {
                    // Copy calibration files server-side
                    $dir = "/var/www/html/CalFiles/";
                    $files = [
                        "sc-cal{$freqVal}.csv" => "sc.csv",
                        "lc-cal{$freqVal}.csv" => "lc.csv",
                        "ph-cal{$freqVal}.csv" => "ph.csv",
                    ];
                    foreach ($files as $src => $dst) {
                        $srcPath = $dir . $src;
                        $dstPath = $dir . $dst;
                        if (!copy($srcPath, $dstPath)) {
                            http_response_code(500);
                            echo "Failed to copy $src to $dst";
                            exit;
                        }
                        // Clean the file: remove comments, whitespace, and linefeeds
                        $lines = file($dstPath, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
                        $cleaned = '';
                        foreach ($lines as $line) {
                            $line = preg_replace('/\/\/.*$/', '', $line); // remove comments
                            $line = preg_replace('/\s+/', '', $line);    // remove spaces/tabs
                            $cleaned .= $line;
                        }
                        file_put_contents($dstPath, $cleaned);
                    }
                }//if in_array
                
                // 11,12,13,14 relays values
                $rlyMap = [];   // freq => [b0, b1, b2, b3]
                if (!file_exists($rlyFile)) {
                    error_log("Relay file missing: $rlyFile");
                } else {
                    $fh = fopen($rlyFile, "r");
                    if ($fh === false) {
                        error_log("Failed to open relay file: $rlyFile");
                    } else {
                        while (($line = fgets($fh)) !== false) {
                            $line = preg_replace('/\/\/.*$/', '', $line);  // Remove // comments
                            $line = trim($line);  // Trim whitespace
                            if ($line === '') continue;  // Skip empty lines
                            // Parse CSV
                            $row = str_getcsv($line);
                            if (count($row) < 5) continue;
                            $freqF = (int)$row[0];
                            $bytes = [];
                            for ($i = 1; $i <= 4; $i++) {
                                $bytes[] = intval($row[$i], 0) & 0xFF;
                            }
                            $rlyMap[$freqF] = $bytes;
                        }
                        fclose($fh);
                    }
                }//if file exsists
         
                if (isset($rlyMap[$freqVal])) {
                    [$b0, $b1, $b2, $b3] = $rlyMap[$freqVal];
                    $setsValues[11] = $b0 | 0x80;  // Enforce MSB = 1 (10xxxxxx protocol)
                    $setsValues[12] = $b1 | 0x80;
                    $setsValues[13] = $b2 | 0x80;
                    $setsValues[14] = $b3 | 0x80;
                } else {
                    error_log("No relay mapping for frequency $freqVal");
                }
                break;
            
            case "nCyc":       $setsValues[15] = (int)$val; break;   //number of cycles in the burst
            // 16,17,18,19 not used
            default:
                // ignore unknown fields (or add mapping here)
                break;
        }
    }

    // Maintain verification nibble on element 0 (high nibble = 0xA)
    $setsValues[0] = (((int)$setsValues[0]) & 0x0F) | 0xA0;

    // Write back sets.csv (atomic-ish with LOCK_EX)
    $written = file_put_contents($setsFile, implode(",", $setsValues) . PHP_EOL, LOCK_EX);
    
    //delete any ACK file before looking for it
    if (file_exists($ackFile)) {
        if (!unlink($ackFile)) {
            error_log("Failed to delete ACK file: $ackFile");
        } else {
            //error_log("Deleted ACK file: $ackFile");
        }
    } else {
        error_log("ACK file does not exist: $ackFile");
    }

    // wait up to 500ms for ack
    $timeout = microtime(true) + 0.5;
    while (!file_exists($ackFile)) {
        usleep(10000); // 10 ms poll
        if (microtime(true) > $timeout) {
            echo "ERR";
            exit;
        }
    }
    
    $setsValues[8] = 0x00;  //clear the calFile flag 
    $written = file_put_contents($setsFile, implode(",", $setsValues) . PHP_EOL, LOCK_EX);  // write again

    header('Content-Type: text/plain');
    if ($written === false) {
        echo "ERR";
        exit(0);
    } else {
        echo "OK";
        exit(0);
    }
}

// ---------- If not POST, render HTML UI (use $statusValues for defaults) ----------
$enOut   = isset($statusValues[0]) ? ($statusValues[0] & 0x0F) : 0; // strip verify nibble
$f_level = $statusValues[1] ?? 0;
$f_mode  = $statusValues[2] ?? 72;
$toff    = $statusValues[3] ?? 0;
$ttOn    = $statusValues[4] ?? 0;
$ttOff   = $statusValues[5] ?? 0;
$ttRst   = $statusValues[6] ?? 0;
$cal     = $statusValues[7] ?? 0;
$calFile = $statusValues[8] ?? "";
// 9: PhaseCalValue
// 10: flgtt  (1= currently in treatment) 0= currently waiting for next treatment
// 11: cntBursts MSB
// 12: cntBursts LSB
$freq   = ($statusValues[13]*256) + $statusValues[14];  //timer value
$freq   = round(24000000 / (($freq + 1) * 2));          //compute to frequency
$nCyc   = $statusValues[15] ?? 0;                       //# of cycles in the burst
// 16,17,18,19 not used  
?>

<!doctype html>
<html>
<head>
  <meta charset="utf-8" />
  <title>UREBS Settings (Auto-save)</title>
  <style>
    body { font-family: Arial, Helvetica, sans-serif; font-size: 13px; }
    table { border-collapse: collapse; }
    td { padding: 6px; vertical-align: middle; }
  </style>
</head>
<body>
<h3>UREBS Settings <button type="button" onclick="location.reload()">Refresh</button></h3>

<form id="autoForm">
<table border="0" cellpadding="5" cellspacing="0">

<!-- Enable Output -->
<tr>
  <td>Enable Output:</td>
  <td><label><input type="radio" name="enOut" value="1" <?=($enOut==1?"checked":"")?> onchange="autoSave(this)"> On </label></td>
  <td><label><input type="radio" name="enOut" value="0" <?=($enOut==0?"checked":"")?> onchange="autoSave(this)"> Off </label></td>
</tr>
</table>

<!-- Burst Frequency -->
<table>
<tr>
   <td>Freq(Hz):</td>
   
   <td>
    <?php
    $freq = $freq ?? 265;  //if freq does not exist assign to 265Hz
    $freqOptions = [210, 215, 220, 225, 230, 235, 240, 245, 250, 255, 260, 265];
    foreach ($freqOptions as $f) {
        $checked = ($freq == $f) ? 'checked' : '';
        echo '<label style="margin-right:4px;">';
        echo "<input type=\"radio\" name=\"freq\" value=\"$f\" $checked onchange=\"autoSave(this)\">$f";
        echo '</label>';
    }
    ?>
   </td>
</tr>
</table>

<!-- Number of Cycles in the burst -->
<table>
<tr>
  <td># cycles in Burst</td>
    <td><input type="number" name="nCyc" size="4" maxlength="4" max="30" value="<?=$nCyc?>" onchange="autoSave(this)"></td>  
</tr>
</table>

<!-- Field Level -->
<table>
<tr>
    <td>Field Level (0-5):</td>
    <td><label><input type="radio" name="f_level" value="0" <?= ($f_level == 0 ? "checked" : "") ?> onchange="autoSave(this)" >0.5mT</label></td>
    <td><label><input type="radio" name="f_level" value="1" <?= ($f_level == 1 ? "checked" : "") ?> onchange="autoSave(this)" >1mT</label></td>
    <td><label><input type="radio" name="f_level" value="2" <?= ($f_level == 2 ? "checked" : "") ?> onchange="autoSave(this)" >2mT</label></td>
    <td><label><input type="radio" name="f_level" value="3" <?= ($f_level == 3 ? "checked" : "") ?> onchange="autoSave(this)" >3mT</label></td>
    <td><label><input type="radio" name="f_level" value="4" <?= ($f_level == 4 ? "checked" : "") ?> onchange="autoSave(this)" >4mT</label></td>
    <td><label><input type="radio" name="f_level" value="5" <?= ($f_level == 5 ? "checked" : "") ?> onchange="autoSave(this)" >5mT</label></td>
</tr>
</table>
<table>
<!-- Field Mode -->
<tr>
  <td>Field Mode:</td>
  <td><label><input type="radio" name="f_mode" value="72"  <?=($f_mode==72  ? "checked" : "")?> onclick="autoSave(this)">Horz</label></td>
  <td><label><input type="radio" name="f_mode" value="86"  <?=($f_mode==86  ? "checked" : "")?> onclick="autoSave(this)">Vert</label></td>
  <td><label><input type="radio" name="f_mode" value="82"  <?=($f_mode==82  ? "checked" : "")?> onclick="autoSave(this)">CW Rot</label></td>
  <td><label><input type="radio" name="f_mode" value="114" <?=($f_mode==114 ? "checked" : "")?> onclick="autoSave(this)">CCW Rot</label></td>
  <td><label><input type="radio" name="f_mode" value="49"  <?=($f_mode==49  ? "checked" : "")?> onclick="autoSave(this)">H-V</label></td>
  <td><label><input type="radio" name="f_mode" value="50"  <?=($f_mode==50  ? "checked" : "")?> onclick="autoSave(this)">CW-CCW</label></td>
  <td><label><input type="radio" name="f_mode" value="51"  <?=($f_mode==51  ? "checked" : "")?> onclick="autoSave(this)">H-V-CW-CCW</label></td>
</tr>
</tr>
</table>
<table>
<!-- Toff -->
<tr>
  <td>Toff 120ms*(1-255)</td>
  <td><input type="number" name="toff" size="4" maxlength="4" max="255" value="<?=$toff?>" onchange="autoSave(this)"></td>
  <td><?php $Brt=(($toff*0.23)+0.1); echo $Brt; ?> seconds</td>
</tr>

<!-- Time ON -->
<table>
<tr>
  <td>Schedule: ON(1-255)</td>
  <td><input type="number" name="ttOn" size="4" maxlength="4" max="255" value="<?=$ttOn?>" onchange="autoSave(this)"></td>
  <td><?php $minutes = round($ttOn * $Brt * 256 / 60);  
            $hours = intdiv($minutes, 60);     // integer-safe division
            $mins  = $minutes % 60;            // now safe because it's an int
            echo "{$hours}h {$mins}m "; ?>   
  </td>


<!-- Time OFF -->

  <td> OFF(0-255)</td>
  <td><input type="number" name="ttOff" size="4" maxlength="4" max="255" value="<?=$ttOff?>" onchange="autoSave(this)"></td>
  <td><?php $minutes = round($ttOff * $Brt * 256 / 60);  
            $hours = intdiv($minutes, 60);     // integer-safe division
            $mins  = $minutes % 60;            // now safe because it's an int
            echo "{$hours}h {$mins}m "; ?>   
  </td>


<!-- Restart Schedule -->
  <td>Restart Schedule</td>
  <td>
    <input type="checkbox" name="ttRst" value="1" <?=($ttRst==1?"checked":"")?> onchange="autoReset(this)">
  </td>

</tr>
</table>

<!-- Calibration / Defaults (mutually exclusive) -->
<table>
<tr>
  <td>Calibration:</td>
  <td>
    <label>
      <input type="checkbox" id="calBox" name="cal" value="1" <?=($cal==1?"checked":"")?> onchange="handleCal(this); autoReset(this)"> Start       
    </label>
  </td>


<!-- Cal File -->
<!-- 0 no Xfer request, 1 get cal file from UREBS-->
<!-- 0x80(128) request for Xfer from RPI to UREBS -->
<!--    0x80 copy from x-cal Files to x -->
<!--    0x81 copy from x-default Files to x  save as 0x80-->
    <td><label><input type="checkbox" id="calFile" name="calFile" value="1" <?=($cal==1?"checked":"")?> onchange="handleCal(this); autoReset(this)">Get Cal Files</label></td>
    <!--<td><label><input type="checkbox" id="calFile" name="calFile" value="128" <?=($cal==128?"checked":"")?> onchange="handleDef(this); autoReset(this)"> Send Cal Files</label></td>-->
    <td><label><input type="checkbox" id="calFile" name="calFile" value="129" <?=($cal==129?"checked":"")?> onchange="handleDef(this); autoReset(this)">Send Default Cal Files</label></td>
  </tr>
</table>

</form>

<button type="button" onclick="location.reload()">Refresh</button>

<pre style='font-size: 12px; font-family: Arial, Helvetica, sans-serif; line-height: 1.2;'>
Default Cal files: are the files you transfer just before doing a calibration.
  -they are copied from the CalFiles directory *-defaults.csv to the *.csv file, then to the UREBS
Get Cal Files: just gets the currently in use files and stores them in the RamDisk.
Send Cal Files: takes the *-cal.csv files from the CalFiles directory, copys them to the *.csv files in same directory, then to the UREBS.
</pre>



<pre style='font-size: 12px; font-family: Arial, Helvetica, sans-serif; line-height: 1.2;'>
Treatment on and off times can be calculated as follows:
Burst Repeat time(BRT) = tOff*120ms + ~100ms
Treatment Time = TreatmentTime*BRT*256

Treatment Time OFF: 0(always on) - 255 (shuts off)


</pre>

<a href="./index.php">Home Page</a>


<script>
/*
 * autoSave: send single changed input to the same PHP page (POST)
 * autoSaveCheckbox: handles checkboxes sending 0 when unchecked
 * handleCal / handleDef: enforce mutual-exclusion and submit proper values
 */

function autoSave(el) {
    const fd = new FormData();
    fd.append(el.name, el.value);

    fetch(window.location.href, { method: "POST", body: fd })
      .then(r => r.text())
      .then(txt => { /* optional: show saved indicator */ })
      .catch(err => console.error("Save error:", err));
}

function autoSaveCheckbox(el) {
    const fd = new FormData();
    fd.append(el.name, el.checked ? el.value : 0);

    fetch(window.location.href, { method: "POST", body: fd })
      .then(r => r.text())
      .catch(err => console.error("Save error:", err));
}

function handleCal(el) {
    if (el.checked) {
        // deselect defaults
        const defBox = document.getElementById("defBox");
        if (defBox) defBox.checked = false;
        autoSave(el); // sends cal=1
    } else {
        autoSaveCheckbox(el); // sends cal=0
    }
}
function handleDef(el) {
    if (el.checked) {
        const calBox = document.getElementById("calBox");
        if (calBox) calBox.checked = false;
        autoSave(el); // sends cal=99
    } else {
        autoSaveCheckbox(el); // sends cal=0
    }
}



function autoReset(el) {
    // special handling if calFile = 0x80
    if (el.value == 0x80) {
        fetch("copyCalFiles.php")
            .then(r => r.text())
            .then(txt => console.log("Copy result:", txt))
            .catch(err => console.error("Copy error:", err));
    }
        // special handling if calFile = 0x81
    if (el.value == 0x81) {
        fetch("copyCalDefaultFiles.php")
            .then(r => r.text())
            .then(txt => console.log("Copy result:", txt))
            .catch(err => console.error("Copy error:", err));
        el.value = 0x80;  //restore to trigger the transfer  
    }
    
    // immediately send current value
    if(el.type === 'checkbox') {
        autoSaveCheckbox(el);
    } else {
        autoSave(el);
    }

    // schedule reset to 0 after 500ms
    setTimeout(function(){
        if(el.type === 'checkbox') {
            el.checked = false;
            autoSaveCheckbox(el);
        } else {
            el.value = 0;
            autoSave(el);
        }
    }, 500);
}


//setInterval(function() {
//    window.location.reload();
//}, 1000); // 1000 milliseconds = 1 second

</script>

</body>
</html>
