// g++ -o urebs urebs.cpp -lgpiod
// /var/www/html/urebs
// sudo -u www-data /var/www/html/urebs

// sudo apt-get install gpiod libgpiod-dev

//Program monitors the urebs for rising edge of gpio4 (P2.2)
//RPi is I2C master and urebs is the slave.
//upon rising edge the RPI gets 80 bytes of status info and saves it to status.csv
//then the RPi sends 20 bytes of setting info (sets.csv)
//   -this program does not create sets (it is created by php code)

//program depends on ramdisk
//sudo nano /etc/fstab        and add
//tmpfs   /var/www/html/ramdisk   tmpfs   size=2M   0   0

#include <sys/stat.h>
#include <iostream>
#include <fstream>
#include <gpiod.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <linux/i2c-dev.h>
#include <vector>
#include <sstream>
#include <string>
#include <poll.h>
#include <sys/types.h>  //for checking if program already running
#include <dirent.h>		//for checking if program already running
#include <errno.h>		//for checking if program already running

using namespace std;

#define I2C_DEV "/dev/i2c-1"
#define SLAVE_ADDR 0x71
#define GPIO_CHIP "/dev/gpiochip0"

uint8_t rxBuf[80];
uint8_t txBuf[20];
uint8_t calBuf[255];
int i2c_fd;

string command_file = "/var/www/html/ramdisk/gpio17.txt";

// ***************************** SUBROUTINES DECLARATIONS **************************
int getProcCntByName(string procName);	// **** CHECK IF PROCESS ALREADY RUNNING ******
int urebsHB(void);      //regular interrupt to get status and send settings to the UREBS
int getCalFiles(void);   //get the Calibration Files
int sendCalFiles(void);  //send the Calibration Files

// *******************************************************************
// ***********************  main *************************************
// *******************************************************************
int main() {
	// test if program is already running
	int cntUREBS = getProcCntByName("urebs");
	if (cntUREBS > 1)
	{
		//then process is already running
		cout << "UREBS Already Running " << endl;
		return 1;  //exit program, 1 indicates failure to run
	}   
    
    gpiod_chip *chip = gpiod_chip_open(GPIO_CHIP);
    if (!chip) {
        cerr << "Failed to open GPIO chip\n";
        return 1;
    }
    
 //       const char *chipname = "gpiochip0";  // or gpiochip4 depending on Pi 5 mapping
 //   int line_num = 17;

    // Open GPIO chip
//    gpiod_chip *chip = gpiod_chip_open_by_name(chipname);
//    if (!chip) {
 //       std::cerr << "Failed to open GPIO chip\n";
//        return 1;
//    }

    // Get GPIO17 line
    gpiod_line *line = gpiod_chip_get_line(chip, 17);
    if (!line) {
        std::cerr << "Failed to get line 17\n";
        gpiod_chip_close(chip);
        return 1;
    }

    // Request output (initial value LOW)
    if (gpiod_line_request_output(line, "set_gpio17", 0) < 0) {
        std::cerr << "Failed to request line as output\n";
        gpiod_chip_close(chip);
        return 1;
    }
    cout << "GPIO17 set LOW\n";
    
    // --- Read back the value ---
    int val = gpiod_line_get_value(line);
    if (val < 0) {
        std::cerr << "Failed to read line value\n";
    } else {
        std::cout << "GPIO17 current value = " << val << std::endl;
    }

    gpiod_line *line4  = gpiod_chip_get_line(chip, 4);
    gpiod_line *line27 = gpiod_chip_get_line(chip, 27);
    gpiod_line *line22 = gpiod_chip_get_line(chip, 22);
    if (!line4 || !line27 || !line22) {
        cerr << "Failed to get one or more GPIO lines\n";
        return 1;
    }

 
    struct gpiod_line_request_config cfg;
    cfg.consumer = "i2c_trigger";
    cfg.request_type = GPIOD_LINE_REQUEST_EVENT_RISING_EDGE;
    cfg.flags = GPIOD_LINE_REQUEST_FLAG_BIAS_PULL_DOWN; // enable internal pull-down

    if (gpiod_line_request(line4, &cfg, 0) < 0 ||
        gpiod_line_request(line27, &cfg, 0) < 0 ||
        gpiod_line_request(line22, &cfg, 0) < 0) {
        cerr << "Failed to request GPIO events with pull-down\n";
        return 1;
    }


    i2c_fd = open(I2C_DEV, O_RDWR);
    if (i2c_fd < 0) {
        cerr << "Failed to open I2C device\n";
        return 1;
    }
    if (ioctl(i2c_fd, I2C_SLAVE, SLAVE_ADDR) < 0) {
        cerr << "Failed to set I2C slave address\n";
        close(i2c_fd);
        return 1;
    }

    cout << "Waiting for GPIO triggers on 4,27,22...\n";

    struct pollfd pfds[3];
    pfds[0].fd = gpiod_line_event_get_fd(line4);
    pfds[0].events = POLLIN;
    pfds[1].fd = gpiod_line_event_get_fd(line27);
    pfds[1].events = POLLIN;
    pfds[2].fd = gpiod_line_event_get_fd(line22);
    pfds[2].events = POLLIN;

    struct gpiod_line_event event;

    while (true) {
        int ret = poll(pfds, 3, 1000); // wait max of 1 seconds so reset gets checked every 2 seconds.
        if (ret < 0) {
            perror("poll");
            break;
        }
        //********************** GPIO 4 ***************************
        // regular interrupt for status and settings
        if (pfds[0].revents & POLLIN) {
            gpiod_line_event_read(line4, &event);
            cout << "GPIO4 rising edge detected → do I2C transaction\n";
            urebsHB();    //get status, and update settings
        }//gpio4


         //********************** GPIO 22 ***************************
         //interrupt to get cal files from the UREBS
        if (pfds[2].revents & POLLIN) {
            gpiod_line_event_read(line22, &event);
            cout << "GPIO22 rising edge detected → ACTION C\n";
           // GPIO went HIGH → read I2C
           //------------------- sc file ----------------  
           // 1st deglitch the line by reading it manually
           usleep(100);
           int val = gpiod_line_get_value(line22);
            if (val < 0) {
                perror("gpiod_line_get_value");
            } else if (val == 1) {
                cout << "GPIO22 confirmed HIGH, proceeding with I2C\n";                
                getCalFiles();
            } else {
                cout << "GPIO22 was LOW after event → glitch ignored\n";
            }
        }
        
        //********************** GPIO 27 ***************************
        //interrupt to send the cal files to the UREBS
        if (pfds[1].revents & POLLIN) {   
            gpiod_line_event_read(line27, &event);
            cout << "GPIO27 rising edge detected → WRITE CalFiles to I2C\n";

            // Debounce/deglitch check
            usleep(100);
            int val = gpiod_line_get_value(line27);
            if (val < 0) {
                perror("gpiod_line_get_value");
            } else if (val == 1) {
                cout << "GPIO27 confirmed HIGH, proceeding with I2C write\n";
                sendCalFiles();
            } else {
                cout << "GPIO27 was LOW after event → glitch ignored\n";
            }    
        }

        //check gpio17.txt file to see the UREBS needs reseting via GPIO17.
        struct stat buffer;
        if (stat(command_file.c_str(), &buffer) == 0) {  //file exists
            gpiod_line_set_value(line, 1);       // set HIGH
            std::cout << "GPIO17 HIGH\n";
            unlink(command_file.c_str());                 // delete file
            usleep(2000000);  //2 second sleep
            gpiod_line_set_value(line, 0);       // default LOW
        }




    }//loop forever

    close(i2c_fd);
    gpiod_chip_close(chip);
    return 0;
}//main


// *******************************************************************
// *********************** SUBROUTINES *******************************
// *******************************************************************

// ********************** urebsHB *************************
// ***** get status and update settings *******************
int urebsHB(void){
   // GPIO went HIGH → read I2C
    if (read(i2c_fd, rxBuf, 80) == 80) {
        cout << "Received: ";
        for (int i = 0; i < 80; i++) {
            cout << static_cast<int>(rxBuf[i]) << " ";
        }
        cout << dec << endl;
    } else {
        cerr << "I2C read error\n";
    }
    
    //also write the rxBuf to a file
    ofstream outFile("/var/www/html/ramdisk/status.csv", ios::out | ios::trunc);
    if (!outFile.is_open()) {
        cerr << "Failed to open status.csv for writing" << endl;
        return 1;
    }
    // write buffer to file
    for (int i =0; i<80; i++) {
        outFile << static_cast<int>(rxBuf[i]);
        if (i<80-1) outFile << ",";
    }
    outFile << "\n";
    if (!outFile) {
        cerr << "Error: write failed " << endl;
        return 1;
    }
    outFile.close();  //close the csv file
    
    //read the sets.csv
    ifstream inFile("/var/www/html/ramdisk/sets.csv");
    if (!inFile.is_open()) {
        cerr << "Warning: sets.csv not found, using default values" << endl;
        // fallback: fill txBuf with zeros
        fill(begin(txBuf), end(txBuf), 0);
    }
    string line;
    if (getline(inFile, line)) {
        stringstream ss(line);
        string value;
        int index = 0;
        while (getline(ss, value, ',') && index < 20) {
            try {
                txBuf[index] = static_cast<uint8_t>(stoi(value));
            } catch (...) {
                txBuf[index] = 0;  // fallback if conversion fails
            }
            index++;
        }
        // pad with 0s if fewer than 20 values
        while (index < 20) {
            txBuf[index++] = 0;
        }
    }
    inFile.close();  
    //create the blank ACK file
    ofstream ack("/var/www/html/ramdisk/sets.ack");
    ack.close();
     
    // --- Print txBuf contents ---
    cout << "txBuf contents: " ;
    for (int i = 0; i < 20; i++) {
        cout <<  (int)txBuf[i] << ',';
    } 
    cout << endl;              
    
    //step 2: send 20 bytes over I2C
    int w= write(i2c_fd, txBuf, 20);
    if (w !=20) {
        perror("write");
    } else {
        cout << "Sent 20 bytes to the MSP" << endl;
    }
    
    return 0;
}



// ********************** getCalFiles *************************
// ***** get the Calibration files from the UREBS *******************
int getCalFiles(void){
       // Do the I2C read/write            
       if (read(i2c_fd, calBuf, 234) == 234) {
            cout << "Received: ";
            for (int i = 0; i < 234; i++) {
                cout << static_cast<int>(calBuf[i]) << " ";
            }
            cout << dec << endl;
        } else {
            cerr << "sc I2C read error\n";
        }
        
        //also write the calBuf to a file
        ofstream outFile1("/var/www/html/ramdisk/sc.csv", ios::out | ios::trunc);
        if (!outFile1.is_open()) {
            cerr << "Failed to open sc.csv for writing" << endl;
            return 1;
        }
        // write buffer to file
        for (int i =0; i<234; i++) {
            outFile1 << static_cast<int>(calBuf[i]);
            if (i<234-1) outFile1 << ",";
        }
        outFile1 << "\n";
        if (!outFile1) {
            cerr << "Error: sc write failed " << endl;
            return 1;
        }
        outFile1.close();  //close the csv file
        
        //------------------- lc file ----------------  
        usleep(3000);
       if (read(i2c_fd, calBuf, 234) == 234) {
            cout << "Received: ";
            for (int i = 0; i < 234; i++) {
                cout << static_cast<int>(calBuf[i]) << " ";
            }
            cout << dec << endl;
        } else {
            cerr << "lc I2C read error\n";
        }
        
        //also write the calBuf to a file
        ofstream outFile2("/var/www/html/ramdisk/lc.csv", ios::out | ios::trunc);
        if (!outFile2.is_open()) {
            cerr << "Failed to open lc.csv for writing" << endl;
            return 1;
        }
        // write buffer to file
        for (int i =0; i<234; i++) {
            outFile2 << static_cast<int>(calBuf[i]);
            if (i<234-1) outFile2 << ",";
        }
        outFile2 << "\n";
        if (!outFile2) {
            cerr << "Error: lc write failed " << endl;
            return 1;
        }
        outFile2.close();  //close the csv file
        
        //------------------- ph file ----------------  
        usleep(3000);
       if (read(i2c_fd, calBuf, 72) == 72) {
            cout << "Received: ";
            for (int i = 0; i < 72; i++) {
                cout << static_cast<int>(calBuf[i]) << " ";
            }
            cout << dec << endl;
        } else {
            cerr << "ph I2C read error\n";
        }
        
        //also write the calBuf to a file
        ofstream outFile3("/var/www/html/ramdisk/ph.csv", ios::out | ios::trunc);
        if (!outFile3.is_open()) {
            cerr << "Failed to open ph.csv for writing" << endl;
            return 1;
        }
        // write buffer to file
        for (int i =0; i<72; i++) {
            outFile3 << static_cast<int>(calBuf[i]);
            if (i<72-1) outFile3 << ",";
        }
        outFile3 << "\n";
        if (!outFile3) {
            cerr << "Error: ph write failed " << endl;
            return 1;
        }
        outFile3.close();  //close the csv file
    


    return 0;
}


// ********************** sendCalFiles *************************
// ***** send the Calibration Files to the UREBS *******************
int sendCalFiles(void){
    // ---------- sc.csv ----------
    {
        ifstream inFile("/var/www/html/CalFiles/sc.csv");
        if (!inFile.is_open()) {
            cerr << "Failed to open sc.csv for reading" << endl;
            return 1;
        }
        vector<uint8_t> buf;
        string token;
        while (getline(inFile, token, ',')) {
            buf.push_back(static_cast<uint8_t>(stoi(token)));
        }
        inFile.close();

        if (buf.size() >= 234) {
            if (write(i2c_fd, buf.data(), 234) != 234) {
                cerr << "sc I2C write error\n";
            } else {
                cout << "sc.csv → I2C OK (" << buf.size() << " bytes loaded, 234 sent)\n";
            }
        } else {
            cerr << "sc.csv does not contain enough data\n";
        }
    }

    // ---------- lc.csv ----------
    usleep(10000);  //10ms delay allows for MCU time to xfer the cal file
    {
        ifstream inFile("/var/www/html/CalFiles/lc.csv");
        if (!inFile.is_open()) {
            cerr << "Failed to open lc.csv for reading" << endl;
            return 1;
        }
        vector<uint8_t> buf;
        string token;
        while (getline(inFile, token, ',')) {
            buf.push_back(static_cast<uint8_t>(stoi(token)));
        }
        inFile.close();

        if (buf.size() >= 234) {
            if (write(i2c_fd, buf.data(), 234) != 234) {
                cerr << "lc I2C write error\n";
            } else {
                cout << "lc.csv → I2C OK (" << buf.size() << " bytes loaded, 234 sent)\n";
            }
        } else {
            cerr << "lc.csv does not contain enough data\n";
        }
    }

    // ---------- ph.csv ----------
    usleep(10000);  //10ms delay allows for MCU time to xfer the cal file
    {
        ifstream inFile("/var/www/html/CalFiles/ph.csv");
        if (!inFile.is_open()) {
            cerr << "Failed to open ph.csv for reading" << endl;
            return 1;
        }
        vector<uint8_t> buf;
        string token;
        while (getline(inFile, token, ',')) {
            buf.push_back(static_cast<uint8_t>(stoi(token)));
        }
        inFile.close();

        if (buf.size() >= 72) {
            if (write(i2c_fd, buf.data(), 72) != 72) {
                cerr << "ph I2C write error\n";
            } else {
                cout << "ph.csv → I2C OK (" << buf.size() << " bytes loaded, 72 sent)\n";
            }
        } else {
            cerr << "ph.csv does not contain enough data\n";
        }
    }
    
    // ---------- calTrg.csv ----------
    // this is the target values used by the calibration routine
    usleep(10000);  //10ms delay allows for MCU time to xfer the cal file
    {
        ifstream inFile("/var/www/html/CalFiles/calTrg.csv");
        if (!inFile.is_open()) {
            cerr << "Failed to open calTrg.csv for reading" << endl;
            return 1;
        }
        vector<uint8_t> buf;
        string token;
        while (getline(inFile, token, ',')) {
            buf.push_back(static_cast<uint8_t>(stoi(token)));
        }
        inFile.close();

        if (buf.size() >= 36) {
            if (write(i2c_fd, buf.data(), 36) != 36) {
                cerr << "calTrg I2C write error\n";
            } else {
                cout << "calTrg.csv → I2C OK (" << buf.size() << " bytes loaded, 36 sent)\n";
            }
        } else {
            cerr << "calTrg.csv does not contain enough data\n";
        }
    }
    
    

       
        
  
    return 0;  
}

// *******************************************************************
// **** CHECK IF PROCESS ALREADY RUNNING ******
int getProcCntByName(string procName)
{
    int pid = -1;
    int cntP = 0;   //process count

    // Open the /proc directory
    DIR *dp = opendir("/proc");
    if (dp != NULL)
    {
        // Enumerate all entries in directory until process found
        struct dirent *dirp;
        while ((dirp = readdir(dp)))
        {
            // Skip non-numeric entries
            int id = atoi(dirp->d_name);
            if (id > 0)
            {
                // Read contents of virtual /proc/{pid}/cmdline file
                string cmdPath = string("/proc/") + dirp->d_name + "/cmdline";
                ifstream cmdFile(cmdPath.c_str());
                string cmdLine;
                getline(cmdFile, cmdLine);
                if (!cmdLine.empty())
                {
                    // Keep first cmdline item which contains the program path
                    size_t pos = cmdLine.find('\0');
                    if (pos != string::npos)
                        cmdLine = cmdLine.substr(0, pos);
                    // Keep program name only, removing the path
                    pos = cmdLine.rfind('/');
                    if (pos != string::npos)
                        cmdLine = cmdLine.substr(pos + 1);
                    // Compare against requested process name
                    if (procName == cmdLine)
                    {
                        pid = id;
                        cout << "pid=" << pid << endl;
                        cntP++;    //increment process count
					}
                }
            }
        }
    }

    closedir(dp);

    return cntP;   //return total number of matches
}
